/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the reusable ccl java library
 * (http://www.kclee.com/clemens/java/ccl/).
 *
 * The Initial Developer of the Original Code is
 * Chr. Clemens Lee.
 * Portions created by Chr. Clemens Lee are Copyright (C) 2002
 * Chr. Clemens Lee. All Rights Reserved.
 *
 * Contributor(s): Chr. Clemens Lee <clemens@kclee.com>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

package ccl.jcf;

import java.io.BufferedInputStream;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.InputStream;
import java.io.IOException;
import java.util.*;
import java.util.zip.*;
import java.awt.Frame;

import lti.java.jcf.*;

import ccl.swing.SwingUtil;
import ccl.util.*;

/**
 * An utility class to work with java byte code files.
 * It uses Matt Yourst's Java Class File library.<p>
 *
 * The highlight of this class is the getMethodCallees method.
 * It returns all methods which are invoked inside a given
 * java method and java class.<br>
 * Another highlight is the 'jwhich' method which does for
 * java classes, what 'which' does for executables on unix.<p>
 *
 * Also provided are methods to find java class files on the
 * file system or inside jar archives.<p>
 *
 * @author  <a href="http://mats.gmd.de/clemens/">Chr. Clemens Lahme</a> (<a href="mailto:clemens.lahme@mailexcite.com"><i>Clemens.Lahme@mailexcite.com</i></a>)
 * @version $Id: JCFUtil.java,v 1.9 2002/10/06 16:41:11 clemens Exp clemens $
 */
public class JCFUtil 
{
    private static final int[] aOpcodeLength = {
        1, // nop
        1, // aconst_null
        1, // iconst_m1
        1, // iconst_0
        1, // iconst_1
        1, // iconst_2
        1, // iconst_3
        1, // iconst_4
        1, // iconst_5
        1, // lconst_0
        1, // lconst_1
        1, // fconst_0
        1, // fconst_1
        1, // fconst_2
        1, // dconst_0
        1, // dconst_1
        2, // bipush
        3, // sipush
        2, // ldc
        3, // ldc_w
        3, // ldc2_w
        2, // iload
        2, // lload
        2, // fload
        2, // dload
        2, // aload
        1, // iload_0
        1, // iload_1
        1, // iload_2
        1, // iload_3
        1, // lload_0
        1, // lload_1
        1, // lload_2
        1, // lload_3
        1, // fload_0
        1, // fload_1
        1, // fload_2
        1, // fload_3
        1, // dload_0
        1, // dload_1
        1, // dload_2
        1, // dload_3
        1, // aload_0
        1, // aload_1
        1, // aload_2
        1, // aload_3
        1, // iaload
        1, // laload
        1, // faload
        1, // daload
        1, // aaload
        1, // baload
        1, // caload
        1, // saload
        2, // istore
        2, // lstore
        2, // fstore
        2, // dstore
        2, // astore
        1, // istore_0
        1, // istore_1
        1, // istore_2
        1, // istore_3
        1, // lstore_0
        1, // lstore_1
        1, // lstore_2
        1, // lstore_3
        1, // fstore_0
        1, // fstore_1
        1, // fstore_2
        1, // fstore_3
        1, // dstore_0
        1, // dstore_1
        1, // dstore_2
        1, // dstore_3
        1, // astore_0
        1, // astore_1
        1, // astore_2
        1, // astore_3
        1, // iastore
        1, // lastore
        1, // fastore
        1, // dastore
        1, // aastore
        1, // bastore
        1, // castore
        1, // sastore
        1, // pop
        1, // pop2
        1, // dup
        1, // dup_x1
        1, // dup_x2
        1, // dup2
        1, // dup2_x1
        1, // dup2_x2
        1, // swap
        1, // iadd
        1, // ladd
        1, // fadd
        1, // dadd
        1, // isub
        1, // lsub
        1, // fsub
        1, // dsub
        1, // imul
        1, // lmul
        1, // fmul
        1, // dmul
        1, // idiv
        1, // ldiv
        1, // fdiv
        1, // ddiv
        1, // irem
        1, // lrem
        1, // frem
        1, // drem
        1, // ineg
        1, // lneg
        1, // fneg
        1, // dneg
        1, // ishl
        1, // lshl
        1, // ishr
        1, // lshr
        1, // iushr
        1, // lushr
        1, // iand
        1, // land
        1, // ior
        1, // lor
        1, // ixor
        1, // lxor
        3, // iinc
        1, // i2l
        1, // i2f
        1, // i2d
        1, // l2i
        1, // l2f
        1, // l2d
        1, // f2i
        1, // f2l
        1, // f2d
        1, // d2i
        1, // d2l
        1, // d2f
        1, // i2b
        1, // i2c
        1, // i2s
        1, // lcmp
        1, // fcmpl
        1, // fcmpg
        1, // dcmpl
        1, // dcmpg
        3, // ifeq
        3, // ifne
        3, // iflt
        3, // ifge
        3, // ifgt
        3, // ifle
        3, // if_icmpeq
        3, // if_icmpne
        3, // if_icmplt
        3, // if_icmpge
        3, // if_icmpgt
        3, // if_icmple
        3, // if_acmpeq
        3, // if_acmpne
        3, // goto
        3, // jsr
        2, // ret
        1, // tableswitch
        1, // lookupswitch
        1, // ireturn
        1, // lreturn
        1, // freturn
        1, // dreturn
        1, // areturn
        1, // return
        3, // getstatic
        3, // putstatic
        3, // getfield
        3, // putfield
        3, // invokevirtual
        3, // invokespecial
        3, // invokestatic
        5, // invokeinterface
        1, // xxxunusedxxx
        3, // new
        2, // newarray
        3, // anewarray
        1, // arraylength
        1, // athrow
        3, // checkcast
        3, // instanceof
        1, // monitorenter
        1, // monitorexit
        4, // wide
        4, // multianewarray
        3, // ifnull
        3, // ifnonnull
        5, // goto_w
        5, // jsr_w
        1, // breakpoint
        1, // 
        1, // 
        1, // 
        1, // 
        1, // 
        1, // 
        1, // 
        1, // 210
        1, // 
        1, // 
        1, // 
        1, // 
        1, // 
        1, // 
        1, // 
        1, // 
        1, // 
        1, // 220
        1, // 
        1, // 
        1, // 
        1, // 
        1, // 
        1, // 
        1, // 
        1, // 
        1, // 
        1, // 230
        1, // 
        1, // 
        1, // 
        1, // 
        1, // 
        1, // 
        1, // 
        1, // 
        1, // 
        1, // 240
        1, // 
        1, // 
        1, // 
        1, // 
        1, // 
        1, // 
        1, // 
        1, // 
        1, // 
        1, // 250
        1, // 
        1, // 
        1, // 
        1, // impdep1
        1  // impdep2
    };

    private static final String[] asOpcode = { 
        "nop",
        "aconst_null",
        "iconst_m1",
        "iconst_0",
        "iconst_1",
        "iconst_2",
        "iconst_3",
        "iconst_4",
        "iconst_5",
        "lconst_0",
        "lconst_1",
        "fconst_0",
        "fconst_1",
        "fconst_2",
        "dconst_0",
        "dconst_1",
        "bipush",
        "sipush",
        "ldc",
        "ldc_w",
        "ldc2_w",
        "iload",
        "lload",
        "fload",
        "dload",
        "aload",
        "iload_0",
        "iload_1",
        "iload_2",
        "iload_3",
        "lload_0",
        "lload_1",
        "lload_2",
        "lload_3",
        "fload_0",
        "fload_1",
        "fload_2",
        "fload_3",
        "dload_0",
        "dload_1",
        "dload_2",
        "dload_3",
        "aload_0",
        "aload_1",
        "aload_2",
        "aload_3",
        "iaload",
        "laload",
        "faload",
        "daload",
        "aaload",
        "baload",
        "caload",
        "saload",
        "istore",
        "lstore",
        "fstore",
        "dstore",
        "astore",
        "istore_0",
        "istore_1",
        "istore_2",
        "istore_3",
        "lstore_0",
        "lstore_1",
        "lstore_2",
        "lstore_3",
        "fstore_0",
        "fstore_1",
        "fstore_2",
        "fstore_3",
        "dstore_0",
        "dstore_1",
        "dstore_2",
        "dstore_3",
        "astore_0",
        "astore_1",
        "astore_2",
        "astore_3",
        "iastore",
        "lastore",
        "fastore",
        "dastore",
        "aastore",
        "bastore",
        "castore",
        "sastore",
        "pop",
        "pop2",
        "dup",
        "dup_x1",
        "dup_x2",
        "dup2",
        "dup2_x1",
        "dup2_x2",
        "swap",
        "iadd",
        "ladd",
        "fadd",
        "dadd",
        "isub",
        "lsub",
        "fsub",
        "dsub",
        "imul",
        "lmul",
        "fmul",
        "dmul",
        "idiv",
        "ldiv",
        "fdiv",
        "ddiv",
        "irem",
        "lrem",
        "frem",
        "drem",
        "ineg",
        "lneg",
        "fneg",
        "dneg",
        "ishl",
        "lshl",
        "ishr",
        "lshr",
        "iushr",
        "lushr",
        "iand",
        "land",
        "ior",
        "lor",
        "ixor",
        "lxor",
        "iinc",
        "i2l",
        "i2f",
        "i2d",
        "l2i",
        "l2f",
        "l2d",
        "f2i",
        "f2l",
        "f2d",
        "d2i",
        "d2l",
        "d2f",
        "i2b",
        "i2c",
        "i2s",
        "lcmp",
        "fcmpl",
        "fcmpg",
        "dcmpl",
        "dcmpg",
        "ifeq",
        "ifne",
        "iflt",
        "ifge",
        "ifgt",
        "ifle",
        "if_icmpeq",
        "if_icmpne",
        "if_icmplt",
        "if_icmpge",
        "if_icmpgt",
        "if_icmple",
        "if_acmpeq",
        "if_acmpne",
        "goto",
        "jsr",
        "ret",
        "tableswitch",
        "lookupswitch",
        "ireturn",
        "lreturn",
        "freturn",
        "dreturn",
        "areturn",
        "return",
        "getstatic",
        "putstatic",
        "getfield",
        "putfield",
        "invokevirtual",
        "invokespecial",
        "invokestatic",
        "invokeinterface",
        "xxxunusedxxx",
        "new",
        "newarray",
        "anewarray",
        "arraylength",
        "athrow",
        "checkcast",
        "instanceof",
        "monitorenter",
        "monitorexit",
        "wide",
        "multianewarray",
        "ifnull",
        "ifnonnull",
        "goto_w",
        "jsr_w",
        "breakpoint",
        "",
        "",
        "",
        "",
        "",
        "",
        "",
        "210",
        "",
        "",
        "",
        "",
        "",
        "",
        "",
        "",
        "",
        "220",
        "",
        "",
        "",
        "",
        "",
        "",
        "",
        "",
        "",
        "230",
        "",
        "",
        "",
        "",
        "",
        "",
        "",
        "",
        "",
        "240",
        "",
        "",
        "",
        "",
        "",
        "",
        "",
        "",
        "",
        "250",
        "",
        "",
        "",
        "impdep1",
        "impdep2",
    };

    /**
     * Finds the location in the file system of a java class file.
     *
     * @return       null could not be found in the given classpath or
     *               if class is hidden in a jar file. In this case
     *               use 'getJarFileName()'.
     *
     * @deprecated   see ccl.util.ClassPathUtil.
     */
    public static String getAbsoluteClassFileName(String sFullClassName_,
                                                  String sClasspath_)
    {
        return ClassPathUtil.getAbsoluteClassFileName( sFullClassName_, sClasspath_ );
    }

    /**
     * Provide a classname (e.g. ccl.jcf.JCFUtil) and a classpath,
     * and you will get the jar file location which contains
     * this class (e.g. lib/ccl.jar).
     *
     * @exception   IOException    if a file operation failed.
     * @exception   ZipException   if any zip operation failed.
     *
     * @deprecated   see ccl.util.ClassPathUtil.
     */
    public static String getJarFileName( String sFullClassName_ )
        throws IOException, ZipException
    {
        return ClassPathUtil.getJarFileName( sFullClassName_ );
    }

    /**
     * Provide a classname (e.g. ccl.jcf.JCFUtil) and a classpath,
     * and you will get the jar file location which contains
     * this class (e.g. lib/ccl.jar).
     *
     * @exception   IOException    if a file operation failed.
     * @exception   ZipException   if any zip operation failed.
     *
     * @deprecated   see ccl.util.ClassPathUtil.
     */
    public static String getJarFileName( String sFullClassName_, 
                                         String sClasspath_ )
        throws IOException, ZipException
    {
        return ClassPathUtil.getJarFileName( sFullClassName_, sClasspath_ );
    }

    /**
     * Load a class into an JcfClassFile object. Then you can
     * inspect this class with the help of the jcf library.
     *
     * @exception   IOException    if a file operation failed.
     * @exception   ZipException   if any zip operation failed.
     */
    public static JcfClassFile readClassFromJar(String sFullClassName_, 
                                                String sClasspath_)
        throws IOException, ZipException
    {
        String sClassFileName = sFullClassName_.replace('.', '/') + ".class";

        // first get all archives from classpath
        Vector vClasspathElements = Util.stringToLines(sClasspath_,
                                                       File.pathSeparatorChar);
        for(Enumeration eElements = vClasspathElements.elements();
            eElements.hasMoreElements(); )
        {
            String sNextArchiv = (String)eElements.nextElement();
            if ((Util.endsWith(sNextArchiv, ".zip") ||
                 Util.endsWith(sNextArchiv, ".jar")) &&
                FileUtil.existsFile(sNextArchiv))
            {
                                /*try {*/
                ZipFile pZipFile = new ZipFile(sNextArchiv);
                ZipEntry pZipEntry = pZipFile.getEntry(sClassFileName);
                if (pZipEntry != null) 
{
                    JcfClassFile pJcfClassFile = readClassFile(pZipFile, pZipEntry);
                    pZipFile.close();
                                                
                    return pJcfClassFile;
                }
                pZipFile.close();
                                /*} catch(Exception pException) {
                                  }*/
            }
        }

        return null;
    }

    /**
     * Convert a byte array to a JcfClassFile object.
     *
     * @exception   IOException    if a file operation failed.
     */
    public static JcfClassFile readClassFile(byte[] abClass_) 
        throws IOException
    {
        ByteArrayInputStream pInputStream = new ByteArrayInputStream(abClass_);
        JcfClassInputStream pJcfClassInputStream = null;

        pJcfClassInputStream = new JcfClassInputStream
            (new BufferedInputStream
                (pInputStream, Math.min
                 (pInputStream.available(), 32767)));
                
        //
        // Read into JcfClassFile object
        //
        JcfClassFile pJcfClassFile = null;
                
        // Read from class input stream, and create structures.
        pJcfClassFile = new JcfClassFile(pJcfClassInputStream);
        pInputStream.close();
                
        return pJcfClassFile;
    }

    /**
     * Get a JcfClassFile object out of jar or zip file.
     *
     * @exception   IOException    if a file operation failed.
     * @exception   ZipException   if any zip operation failed.
     */
    public static JcfClassFile readClassFile(ZipFile zipOpen_, 
                                             ZipEntry pZipEntry_)
        throws IOException, ZipException
    {
        int compressedSize = (int)pZipEntry_.getCompressedSize();
        int uncompressedSize = (int)pZipEntry_.getSize();
        if (uncompressedSize == -1) 
{
            throw new ZipException();
        }
        byte abtClass[] = new byte[uncompressedSize];
        InputStream pInputStream = zipOpen_.getInputStream(pZipEntry_);

        int bytesRead = 0;
        while(bytesRead < uncompressedSize) 
{
            bytesRead += pInputStream.read(abtClass, bytesRead, uncompressedSize - bytesRead);
        }

        JcfClassFile pJcfClassFile = readClassFile(abtClass);

        return pJcfClassFile;
    }

    /**
     * Either the absolut file name and path is given back or
     * the java archive file name which contains this class is
     * given back.
     *
     * @exception   ClassNotFoundException   if the given class could not be found.
     * @exception   IOException              if a file operation failed.
     * @exception   ZipException   if any zip operation failed.
     *
     * @deprecated   see ccl.util.ClassPathUtil.
     */
    public static String getClassOrJarFileName( String sFullClassName_ )
        throws IOException,
               ClassNotFoundException,
               ZipException
    {
        return ClassPathUtil.getClassOrJarFileName( sFullClassName_ );
    }

    /**
     * Either the absolut file name and path is given back or
     * the java archive file name which contains this class is
     * given back.
     *
     * @exception   ClassNotFoundException   if the given class could not be found.
     * @exception   IOException              if a file operation failed.
     * @exception   ZipException   if any zip operation failed.
     *
     * @deprecated   see ccl.util.ClassPathUtil.
     */
    public static String getClassOrJarFileName( String sFullClassName_,
                                                String sClasspath_ ) 
        throws IOException,
               ClassNotFoundException,
               ZipException
    {
        return ClassPathUtil.getClassOrJarFileName( sFullClassName_, sClasspath_ );
    }

    /**
     * Create a JcfClassFile object reading from the file system.
     *
     * @exception   ClassNotFoundException   if the given class could not be found.
     * @exception   IOException              if a file operation failed.
     * @exception   ZipException   if any zip operation failed.
     */
    public static JcfClassFile readClassFile(String sFullClassName_,
                                             String sClasspath_) 
        throws IOException,
               ClassNotFoundException,
               ZipException
    {
        String sAbsoluteClassFileName = getAbsoluteClassFileName
               (sFullClassName_, sClasspath_);

        if (sAbsoluteClassFileName != null) 
{
            return readClassFile(sAbsoluteClassFileName);
        }

        JcfClassFile pJcfClassFile = readClassFromJar
               (sFullClassName_, sClasspath_);
        if (pJcfClassFile == null) 
{
            throw new ClassNotFoundException();
        }

        return pJcfClassFile;
    }

    /**
     * Create a JcfClassFile object reading from the file system.
     *
     * @exception   IOException    if a file operation failed.
     */
    public static JcfClassFile readClassFile(String sClassFullFileName_) 
        throws IOException
    {
        //
        // Open the class file.
        //
        FileInputStream pFileInputStream = null;
        JcfClassInputStream pJcfClassInputStream = null;

        pFileInputStream = new FileInputStream(sClassFullFileName_);
        pJcfClassInputStream = new JcfClassInputStream
            (new BufferedInputStream
                (pFileInputStream, Math.min
                 (pFileInputStream.available(), 32767)));
                
        //
        // Read into JcfClassFile object
        //
        JcfClassFile pJcfClassFile = null;
                
        // Read from class input stream, and create structures.
        pJcfClassFile = new JcfClassFile(pJcfClassInputStream);
        pFileInputStream.close();
                
        return pJcfClassFile;
    }

    /**
     * Increment the code pointer inside a jav byte code class.
     */
    private static int _getIncrementedCodePointer(int codePointer_,
                                                  byte[] abCode_)
    {
        Util.debug("JCFUtil._getIncrementedCodePointer(..).codePointer_: " + codePointer_);
        Util.debug("JCFUtil._getIncrementedCodePointer(..).abCode_.length: " + abCode_.length);
        int opcode = Util.byteToInt(abCode_[codePointer_]);
        Util.debug("JCFUtil._getIncrementedCodePointer(..).opcode: " + opcode);

        if (opcode == 196) 
{ 
            // iinc
            if (abCode_[codePointer_ + 1] == 132) 
{
                codePointer_ += 6;
            }
        }
 else if (opcode == 170) 
{
            codePointer_ += 4 - (codePointer_ % 4) + 4;
            byte[] abSignedInt = new byte[4];
            abSignedInt[0] = abCode_[codePointer_++];
            abSignedInt[1] = abCode_[codePointer_++];
            abSignedInt[2] = abCode_[codePointer_++];
            abSignedInt[3] = abCode_[codePointer_++];
            int low = Util.bytesToInt(abSignedInt);
            abSignedInt[0] = abCode_[codePointer_++];
            abSignedInt[1] = abCode_[codePointer_++];
            abSignedInt[2] = abCode_[codePointer_++];
            abSignedInt[3] = abCode_[codePointer_++];
            int n = Util.bytesToInt(abSignedInt) - low + 1;
            codePointer_ += n*4;
        }
 else if (opcode == 171) 
{
            codePointer_ += 4 - (codePointer_ % 4) + 4;
            byte[] abSignedInt = new byte[4];
            abSignedInt[0] = abCode_[codePointer_++];
            abSignedInt[1] = abCode_[codePointer_++];
            abSignedInt[2] = abCode_[codePointer_++];
            abSignedInt[3] = abCode_[codePointer_++];
            int n = Util.bytesToInt(abSignedInt);
            codePointer_ += n*8;
        }
 else 
{
            codePointer_ += aOpcodeLength[opcode];
        }
        Util.debug("JCFUtil._getIncrementedCodePointer(..).codePointer_: " + codePointer_);
                
        return codePointer_;
    }

    /**
     * Convert an internal java byte code signature to
     * a format which is more presentable to human users.
     */
    public static String getParameters(String sSignature_) 
    {
        // remove return type, not needed for comparison
        // und ohne Klammern erstmal, diese werden spaeter
        // wieder drangehngt
        String sDescriptors = sSignature_.substring
               (1, sSignature_.lastIndexOf(')'));

        ByteArrayInputStream isParameters = 
               new ByteArrayInputStream( sDescriptors.getBytes() );
        TypeDescriptor pTypeDescriptor = new TypeDescriptor
            (isParameters);
        String sParameters = pTypeDescriptor.parse();
        //isParameters.close();

        return sParameters;
    }

    /**
     * Get all methods which are invoked inside a given method.
     *
     * @param   frmParent_   Needed to report error messages.
     */
    public static Hashtable getMethodCallees(Frame frmParent_,
                                             String sClassFullFileName_,
                                             String sMethod_)
    {
        Hashtable htMethodCallees = new Hashtable();
        Object oShare = new Object();

        // read class file
        JcfClassFile pJcfClassFile = null;
        try 
{
            pJcfClassFile = readClassFile(sClassFullFileName_);
        }
 catch(Exception pException) 
{
            SwingUtil.showMessage( frmParent_, "Error: Exception while reading class file '" + sClassFullFileName_ + "':\n" + pException );
                        
            return htMethodCallees;
        }
        
        JcfConstantPool pJcfConstantPool = pJcfClassFile.
               getConstantPool();

        JcfMemberCollection pJcfMemberCollection
               = pJcfClassFile.getMethods();
        Enumeration eMembers = pJcfMemberCollection.elements();
        String sMethodPrefix = sMethod_.substring(0, sMethod_.indexOf('('));
        while (eMembers.hasMoreElements()) 
{
            JcfMember pJcfMember = (JcfMember)eMembers.nextElement();
            String sMethodName = pJcfConstantPool.utfAt
                   (pJcfMember.mbrNameIndex);
            if (!sMethodPrefix.equals(sMethodName)) 
{
                continue;
            }
            String sMethodSignature = pJcfConstantPool.utfAt
                   (pJcfMember.mbrSignatureIndex);
            String sParameters = getParameters(sMethodSignature);
            if (sMethod_.equals(sMethodName + "(" + sParameters + ")")) 
{
                Util.debug("JCFUtil.getMethodCallees(..).METHOD_FOUND");
                JcfAttributeCollection pJcfAttributeCollection =
                       pJcfMember.getAttributes();
                for(Enumeration eAttributes = pJcfAttributeCollection.elements();
                    eAttributes.hasMoreElements(); )
                {
                    JcfAttribute pJcfAttribute = (JcfAttribute)eAttributes.nextElement();
                    String sAttrname = pJcfConstantPool.utfAt(pJcfAttribute.getNameIndex());
                    if (sAttrname.equals("Code")) 
{
                        JcfCodeAttribute pJcfCodeAttribute = (JcfCodeAttribute)pJcfAttribute;
                        byte[] abCode = pJcfCodeAttribute.catCode;
                        /*for(int i = 0; i < abCode.length; i++) {
                          Util.debug("Jacob.parseMethodCallees(..).abCode[" + i + "]:\t" + Util.btoi(abCode[i]));
                          }*/
                        int codePointer = 0;
                        while(codePointer < abCode.length) 
{
                            int opcode = Util.byteToInt(abCode[codePointer]);
                            Util.debug("JCFUtil.getMethodCallees(..).codePointer: " + codePointer);
                            Util.debug("JCFUtil.getMethodCallees(..).asOpcode[]: " + asOpcode[opcode]);
                                                        
                            // invokevirtual
                            if (182 <= opcode && opcode <= 185) 
{
                                int methodIndex = 256 * Util.byteToInt(abCode[codePointer + 1]) +
                                       Util.byteToInt(abCode[codePointer + 2]);
                                CptMemberOrInterface pCptMemberOrInterface = (CptMemberOrInterface)pJcfConstantPool.
                                       elementAt(methodIndex);
                                int classIndex = pCptMemberOrInterface.classIndex;
                                CptClass pCptClass = (CptClass)pJcfConstantPool.elementAt(classIndex);
                                String sCalleeClassFullName = pJcfConstantPool.utfAt(pCptClass.nameIndex).replace('/', '.');
                                Util.debug("JCFUtil.getMethodCallees(..).sCalleeClassFullName: " + sCalleeClassFullName);
                                                                
                                int nameTypeIndex = pCptMemberOrInterface.nameTypeIndex;
                                CptNameType pCptNameType = (CptNameType)pJcfConstantPool.elementAt(nameTypeIndex);
                                String sCalleeMethodName = pJcfConstantPool.utfAt(pCptNameType.nameIndex);
                                Util.debug("JCFUtil.getMethodCallees(..).sCalleeMethodName: " + sCalleeMethodName);
                                if (opcode == 183 && sCalleeMethodName.equals("<init>")) 
{
                                    sCalleeMethodName = sCalleeClassFullName.substring(sCalleeClassFullName.lastIndexOf('.') + 1);
                                }
                                Util.debug("JCFUtil.getMethodCallees(..).sCalleeMethodName: " + sCalleeMethodName);
                                String sCalleeMethodSignature = pJcfConstantPool.utfAt(pCptNameType.signatureIndex);
                                Util.debug("JCFUtil.getMethodCallees(..).sCalleeMethodSignature: " + sCalleeMethodSignature);
                                String sMethodParameters = getParameters(sCalleeMethodSignature);
                                                                
                                String sCalleeMethod = sCalleeClassFullName + "." +
                                       sCalleeMethodName + "(" + sMethodParameters + ")";
                                Util.debug("JCFUtil.getMethodCallees(..).sCalleeMethod: " + sCalleeMethod);
                                htMethodCallees.put(sCalleeMethod, oShare);
                            }
                                                        
                            // opcodes with variable byte code length
                            codePointer = _getIncrementedCodePointer
                                   (codePointer, abCode);
                        }
                        Util.debug("Jacob.parseMethodCallees(..).codePointer: " + codePointer);
                    }
                }
                                
                break;
            } // if equals parse method
        } // while methods
                
        return htMethodCallees;
    }

    /**
     * Like which for unix, finds the location
     * of a given java class,<br>
     * e.g. jwhich ccl.util.Test<br>
     * -> /home/clemens/bunin/ccl/lib/ccl.jar<br>
     * or<br>
     * -> /home/clemens/bunin/ccl/classes/ccl/util/Test.class .
     *
     * @return   null   if nothing was found.
     *
     * @deprecated   see ccl.util.ClassPathUtil.
     */
    public static String jwhich( String sClassName_,
                                 String sClasspath_ )
    {
        return ClassPathUtil.jwhich( sClassName_, sClasspath_ );
    }

    /**
     * This method searches in the current classpath for the
     * given object. That class should be either in a jar
     * file which is located inside a lib directory or
     * as a pure class file under a classes directory. The
     * directory containing either the classes or the lib
     * directory will be returned as the applicaton home
     * directory. If this is not found, null is returned.
     *                    
     * @param    oClass   An instantiated class which belongs
     *                    to the application.
     *                    
     * @return            null if no home directory was found.
     *
     * @deprecated   see ccl.util.ClassPathUtil.
     */
    public static String getApplicationHome( Object oClass ) 
    {
        return ClassPathUtil.getApplicationHome( oClass );
    }
}
