//
//  JCF Java Class File Manipulation Package (package lti.java.jcf)
//  Copyright (c) 1997 Matt T. Yourst. All rights reserved.
//
//  Version:  1.00.171
//  Date:     January through May 1997
//  Author:   Matt Yourst [yourst@laserstars.com]
//  Language: Java 1.1+
//

package lti.java.jcf;

import java.io.*;
import lti.java.jcf.RuntimeConstants;       

/** The JcfClassFile class is an in-memory representation of the structure
    and format of a Java 1.x class file. JcfClassFile maintains references
    to various objects representing subsections of a class file, and allows
    loading, manipulation, and saving of class files. */

public class JcfClassFile implements RuntimeConstants
{
  /** The 32-bit "magic number" of the class file. Should be
      RuntimeConstants.JAVA_MAGIC (0xcafebabe) */
  public int hdrMagic;
  /** The major version of the class file (should be JAVA_VERSION = 3). */
  public short hdrMajor;
  /** The minor version of the class file (should be JAVA_MINOR_VERSION = 45). */
  public short hdrMinor;
  protected String fullClassName;
    public final String getFullName() { return fullClassName; }
  protected JcfConstantPool constants;
    public final JcfConstantPool getConstantPool() { return constants; }
  public short hdrAccessFlags;
  public short hdrThisClassName;
  public short hdrSuperClassName;
  protected JcfInterfaceCollection interfaces;
    public final JcfInterfaceCollection getInterfaces() { return interfaces; }
  protected JcfMemberCollection fields;
    public final JcfMemberCollection getFields() { return fields; }
  protected JcfMemberCollection methods;
    public final JcfMemberCollection getMethods() { return methods; }
  protected JcfAttributeCollection attributes;
    public final JcfAttributeCollection getAttributes() { return attributes; }

  /** The following readXXXX factory functions instantiate a new instance of
      the corresponding object types by reading them from a JcfClassInput. */
  /** Instantiates a new JcfConstantPool (or a subclass), read from a
      JcfClassInput. Override to return a user-defined subclass of
      JcfConstantPool.
    * @param ist The JcfClassInput that the constant pool is to be read from. */
  public JcfConstantPool readConstantPool(JcfClassInput ist)
    throws IOException
    { return new JcfConstantPool(ist, this); }

  /** Instantiates a new JcfInterfaceCollection (or a subclass), read from a
      JcfClassInput. Override to return a user-defined subclass of
      JcfInterfaceCollection.
    * @param ist The JcfClassInput that the interface array is to be read from. */
  public JcfInterfaceCollection readInterfaces(JcfClassInput ist)
    throws IOException
    { return new JcfInterfaceCollection(ist, constants); }

  /** Instantiates a new JcfMemberCollection (or a subclass), read from a
      JcfClassInput. Override to return a user-defined subclass of
      JcfMemberCollection.
    * @param ist The JcfClassInput that the members are to be read from. */
  public JcfMemberCollection readMemberCollection(JcfClassInput ist)
    throws IOException
    { return new JcfMemberCollection(ist, getConstantPool()); }

  /** Instantiates a new JcfAttributeCollection (or a subclass), read from a
      JcfClassInput. Override to return a user-defined subclass of
      JcfAttributeCollection. <p> This method is called not only to read
      the final "attributes" section of the class file, but also for
      embedded attributes within fields and methods. </p>
    * @param ist The JcfClassInput that the attributes are to be read from. */
  public JcfAttributeCollection readAttributes(JcfClassInput ist)
    throws IOException
    { return new JcfAttributeCollection(ist, getConstantPool()); }

  public JcfClassFile() { super(); }

  /** Instantiates a new JcfClassFile by reading its data from a JcfClassInput.
      The only function of this method is to invoke read(). */
  public JcfClassFile (JcfClassInput ist)
    throws IOException, ClassFormatError
  {
    read(ist);
  }

  /** Returns true if "magic" matches a specific magic header signature value. */
  public boolean isValidMagic(int magic)
  {
    return (magic == JAVA_MAGIC);
  }
  
  /** Initializes a JcfClassFile by reading its data from a JcfClassInput.
      The data read from the JcfClassInput should constitute a valid Java 1.x
      class file. <p>Much of the actual reading of the data is delegated
      to other classes in this package, which are instantiated as the
      corresponding part of the class file is read in.</p> */
  public void read(JcfClassInput ist)
    throws IOException, ClassFormatError
  {
    hdrMagic = ist.readIntVerbatim();
    if (!isValidMagic(hdrMagic))
      throw new ClassFormatError("Invalid magic signature");
    hdrMinor = ist.readShort();
    hdrMajor = ist.readShort();
    constants = readConstantPool(ist);
    hdrAccessFlags = ist.readShortVerbatim();
    hdrThisClassName = ist.readCPRef();
    hdrSuperClassName = ist.readCPRef();
    fullClassName = constants.classNameAt(hdrThisClassName);
    interfaces = readInterfaces(ist);
    fields = readMemberCollection(ist);
    methods = readMemberCollection(ist);
    attributes = readAttributes(ist);
  }

  /** Instantiates a new JcfClassFile by reading its data from a JcfClassInput.
      The data read from the JcfClassInput should constitute a valid Java 1.x
      class file. <p>Much of the actual reading of the data is delegated
      to other classes in this package, which are instantiated as the
      corresponding part of the class file is read in.</p> */

  /** Writes out the (potentially updated) class file to a JcfClassOutput.
      Aster writing header information, the actual writing is delegated
      to the various subobjects making up the JcfClassFile object. */
  public void write (JcfClassOutput ost) throws IOException
  {
    // Write header to compressed file.
    ost.writeIntVerbatim(hdrMagic);
    ost.writeShort(hdrMinor);
    ost.writeShort(hdrMajor);
    getConstantPool().write(ost);
    ost.writeShortVerbatim(hdrAccessFlags);
    ost.writeCPRef(hdrThisClassName);
    ost.writeCPRef(hdrSuperClassName);
    getInterfaces().write(ost);
    getFields().write(ost);
    getMethods().write(ost);
    getAttributes().write(ost);
  }
}
