/* PgSqlClient - ADO.NET Data Provider for PostgreSQL 7.4+
 * Copyright (c) 2003-2004 Carlos Guzman Alvarez
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

using System;
using System.Data;
using System.Text;

namespace PostgreSql.Data.PgSqlClient.DbSchema
{
	internal class PgViewPrivilegesSchema : PgAbstractDbSchema
	{
		#region Constructors

		public PgViewPrivilegesSchema() : base("ViewPrivileges")
		{
		}

		#endregion

		#region Add Methods

		public override void AddTables()
		{
			AddTable("pg_class");
		}

		public override void AddRestrictionColumns()
		{
			AddRestrictionColumn("pg_namespace.nspname", "VIEW_SCHEMA", null);
			AddRestrictionColumn("pg_class.relname", "VIEW_NAME", null);
		}

		public override void AddDataColumns()
		{
			AddDataColumn("pg_class.relacl", "PRIVILEGES");
		}

		public override void AddJoins()
		{
			AddJoin("left join", "pg_namespace", "pg_class.relnamespace = pg_namespace.oid");
			AddJoin("left join", "pg_rewrite", "pg_class.oid = pg_rewrite.ev_class");
		}

		public override void AddOrderByColumns()
		{
			AddOrderBy("pg_namespace.nspname");
			AddOrderBy("pg_class.relname");
		}

		public override void AddWhereFilters()
		{
			AddWhereFilter("pg_class.relkind = 'v'");
		}

		#endregion

		#region Overriden Methods

		public override DataTable GetDbSchemaTable(PgConnection connection, object[] restrictions)
		{
			DataTable tablesSchema = base.GetDbSchemaTable(connection, restrictions);
			DataTable privileges = getPrivilegesDataTable();

			privileges.BeginLoadData();

			foreach (DataRow row in tablesSchema.Rows)
			{
				if (row["PRIVILEGES"] != System.DBNull.Value)
				{
					PgPrivilege[] priv = DecodePrivileges((string[])row["PRIVILEGES"]);

					for (int i = 0; i < priv.Length; i++)
					{
						DataRow newRow = privileges.NewRow();

						newRow["VIEW_SCHEMA"]	= row["VIEW_SCHEMA"];
						newRow["VIEW_NAME"]		= row["VIEW_NAME"];
						newRow["USER_NAME"]		= priv[i].User;

						FillPrivileges(newRow, priv[i].Privileges);

						privileges.Rows.Add(newRow);
					}
				}
			}

			privileges.EndLoadData();

			return privileges;
		}


		private DataTable getPrivilegesDataTable()
		{
			DataTable privileges = new DataTable("Table_Privileges");

			privileges.BeginInit();

			privileges.Columns.Add("VIEW_SCHEMA", Type.GetType("System.String"));
			privileges.Columns.Add("VIEW_NAME", Type.GetType("System.String"));
			privileges.Columns.Add("USER_NAME", Type.GetType("System.String"));

			AddPrivilegesColumns(privileges);

			privileges.EndInit();

			return privileges;
		}

		#endregion

		#region Parse Methods

		public override object[] ParseRestrictions(object[] restrictions)
		{
			object[] parsed = restrictions;

			return parsed;
		}

		#endregion
	}
}