/* PgSqlClient - ADO.NET Data Provider for PostgreSQL 7.4+
 * Copyright (c) 2003-2004 Carlos Guzman Alvarez
 * 
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

using System;
using System.Text;
using System.Collections;
using System.Text.RegularExpressions;

using PostgreSql.Data.NPgClient;

namespace PostgreSql.Data.PgSqlClient
{	
	internal sealed class PgDbConnection : MarshalByRefObject
	{
		#region Fields

		private PgDbClient			db;
		private string				connectionString;
		private PgConnectionParams	settings;
		private long				created;
		private long				lifetime;
		private bool				pooled;
		private Regex				search;

		#endregion

		#region Properties

		public PgDbClient DB
		{
			get { return this.db; }
		}

		public string ConnectionString
		{
			get { return this.connectionString; }
		}

		public long Lifetime
		{
			get { return this.lifetime; }
		}
		public long Created
		{
			get { return this.created; }
			set { this.created = value; }
		}
		
		public bool Pooled
		{
			get { return this.pooled; }
			set { this.pooled = value; }
		}

		public PgConnectionParams Settings
		{
			get { return this.settings; }
		}

		#endregion

		#region Constructors

		private PgDbConnection()
		{
			this.settings			= new PgConnectionParams();
			this.search				= new Regex(@"([\w\s\d]*)\s*=\s*([^;]*)");

			this.connectionString	= String.Empty;
			this.lifetime			= 0;
			this.created			= 0;
			this.pooled				= true;
			this.db					= new PgDbClient();
		}

		public PgDbConnection(string connectionString) : this()
		{
			this.connectionString = connectionString;
			this.parseConnectionString();
		}

		#endregion

		#region Methods

		public void Connect()
		{							
			try
			{
				this.db.Settings = this.Settings;
				this.db.Connect();
			}
			catch (PgClientException ex)
			{
				throw new PgException(ex.Message, ex);
			}
		}
		
		public void Disconnect()
		{	
			try
			{
				this.db.Disconnect();
			}
			catch (PgClientException ex)
			{
				throw new PgException(ex.Message, ex);
			}
		}

		private void parseConnectionString()
		{
			MatchCollection	elements = search.Matches(this.connectionString);

			foreach (Match element in elements)
			{
				if (element.Groups[2].Value.Trim().Length > 0)
				{
					switch (element.Groups[1].Value.Trim().ToLower())
					{
						case "datasource":
						case "server":
						case "host":
							this.settings.ServerName = element.Groups[2].Value.Trim();
							break;

						case "database":
							this.settings.Database = element.Groups[2].Value.Trim();
							break;

						case "user name":
						case "user":
							this.settings.UserName = element.Groups[2].Value.Trim();
							break;

						case "user password":
						case "password":
							this.settings.UserPassword = element.Groups[2].Value.Trim();
							break;

						case "port":
							this.settings.ServerPort = Int32.Parse(element.Groups[2].Value.Trim());
							break;

						case "connection lifetime":
							this.lifetime = Int32.Parse(element.Groups[2].Value.Trim());
							this.lifetime *= TimeSpan.TicksPerSecond;
							break;

						case "timeout":
						case "connection timeout":
							this.settings.Timeout = Int32.Parse(element.Groups[2].Value.Trim());
							break;

						case "packet size":
							this.settings.PacketSize = Int32.Parse(element.Groups[2].Value.Trim());
							break;

						case "pooling":
							this.settings.Pooling = Boolean.Parse(element.Groups[2].Value.Trim());
							break;

						case "ssl":
							this.settings.SSL = Boolean.Parse(element.Groups[2].Value.Trim());
							break;

						case "simple query mode":
							this.settings.SimpleQueryMode = Boolean.Parse(element.Groups[2].Value.Trim());
							break;
					}
				}
			}

			if (settings.UserName == String.Empty || settings.ServerName == String.Empty || settings.ServerPort == 0)
			{
				throw new ArgumentException("An invalid connection string argument has been supplied or a required connection string argument has not been supplied.");
			}
			else
			{
				if (settings.PacketSize < 512 || settings.PacketSize > 32767)
				{
					StringBuilder msg = new StringBuilder();

					msg.AppendFormat("'Packet Size' value of {0} is not valid.\r\nThe value should be an integer >= 512 and <= 32767.", settings.PacketSize);

					throw new ArgumentException(msg.ToString());
				}
			}
		}

		internal bool Verify()
		{
			bool isValid = true;

			try
			{
				// Try to send a Sync message to the PostgreSQL Server
				this.db.Sync();
			}
			catch (Exception)
			{
				isValid = false;
			}

			return isValid;
		}

		#endregion
	}
}